import { NextResponse } from 'next/server'
import { RentIQAnalytics } from '../../../../lib/rentiq-analytics'

const brevo = require('@getbrevo/brevo')
const rentiqAnalytics = RentIQAnalytics.getInstance()

export async function POST(request: Request) {
  try {
    const { date } = await request.json()
    
    // Get RentIQ data
    const rentiqData = await rentiqAnalytics.calculateRentIQ(date)
    
    if (!rentiqData.rentiq_units || rentiqData.rentiq_units.length === 0) {
      return NextResponse.json({
        status: 'error',
        error: 'No RentIQ units found to export'
      }, { status: 400 })
    }

    // Generate CSV content
    const headers = ['Unit', 'Market Rent', 'Suggested Rent', 'Days Vacant', 'Category', 'Min Threshold', 'Cap Applied', 'Pricing Tier']
    const csvContent = [
      headers.join(','),
      ...rentiqData.rentiq_units.map((unit: any) => [
        unit.unit,
        unit.market_rent || 0,
        unit.suggested_new_rent,
        unit.days_vacant,
        unit.assigned_category,
        unit.minimum_threshold,
        unit.cap_applied ? 'Yes' : 'No',
        unit.pricing_tier
      ].join(','))
    ].join('\n')

    // Convert CSV to base64 for email attachment
    const csvBase64 = Buffer.from(csvContent).toString('base64')
    
    // Setup Brevo API
    const apiInstance = new brevo.TransactionalEmailsApi()
    apiInstance.setApiKey(brevo.TransactionalEmailsApiApiKeys.apiKey, process.env.BREVO_API_KEY)

    const todaysDate = new Date().toLocaleDateString('en-US', {
      year: 'numeric',
      month: 'long',
      day: 'numeric'
    })

    const emailData = {
      sender: { email: 'noreply@gardencommand.com', name: 'Garden Command Center' },
      to: [{ email: 'leasing@cynthiagardens.com', name: 'Leasing Team' }],
      subject: `RentIQ Suggestions for ${todaysDate}`,
      htmlContent: `
        <div style="font-family: Arial, sans-serif; max-width: 600px; margin: 0 auto; padding: 20px;">
          <h2 style="color: #4f46e5; margin-bottom: 20px;">🏢 RentIQ Pricing Suggestions</h2>
          
          <p style="margin-bottom: 15px;">Dear Leasing Team,</p>
          
          <p style="margin-bottom: 15px;">Please see attached RentIQ pricing suggestions for <strong>${todaysDate}</strong>.</p>
          
          <div style="background: #f8fafc; border: 1px solid #e2e8f0; border-radius: 8px; padding: 15px; margin: 20px 0;">
            <h3 style="color: #374151; margin: 0 0 10px 0;">📊 Summary:</h3>
            <ul style="margin: 0; padding-left: 20px; color: #4b5563;">
              <li><strong>${rentiqData.rentiq_units.length}</strong> units with pricing suggestions</li>
              <li>Current occupancy: <strong>${rentiqData.current_occupancy.toFixed(1)}%</strong></li>
              <li>Target: <strong>${rentiqData.target_occupancy}%</strong> occupancy</li>
              <li>Units needed: <strong>${rentiqData.units_needed_for_95}</strong> more units</li>
            </ul>
          </div>
          
          <p style="margin-bottom: 15px;">The attached CSV contains detailed pricing recommendations with category mapping and progressive discounting based on vacancy duration.</p>
          
          <p style="margin-bottom: 15px;">Best regards,<br>
          <strong>Garden Command Center</strong><br>
          <em>Automated RentIQ System</em></p>
          
          <hr style="border: none; border-top: 1px solid #e2e8f0; margin: 20px 0;">
          <p style="font-size: 12px; color: #6b7280; margin: 0;">
            This email was automatically generated by the Garden Command Center RentIQ system.
          </p>
        </div>
      `,
      attachment: [
        {
          content: csvBase64,
          name: `rentiq-suggestions-${rentiqData.date}.csv`,
          type: 'text/csv'
        }
      ]
    }

    // Send email
    const emailResult = await apiInstance.sendTransacEmail(emailData)
    
    console.log(`[RENTIQ_EXPORT] Email sent successfully to leasing@cynthiagardens.com - Message ID: ${emailResult.messageId}`)

    return NextResponse.json({
      status: 'success',
      message: 'RentIQ suggestions exported and emailed successfully',
      data: {
        csvContent,
        emailSent: true,
        messageId: emailResult.messageId,
        unitsCount: rentiqData.rentiq_units.length,
        date: rentiqData.date
      }
    })

  } catch (error) {
    console.error('[RENTIQ_EXPORT] Error:', error)
    return NextResponse.json({
      status: 'error',
      error: error instanceof Error ? error.message : 'Failed to export and email RentIQ data'
    }, { status: 500 })
  }
}