import { prisma } from './prisma'

export interface RentIQUnit {
  unit: string
  tenant_status: string
  monthly_rent: number | null
  market_rent: number | null
  unit_type: string | null
  suggested_new_rent: number
  pricing_tier: 'Tier 1' | 'Tier 2' | 'Tier 3'
  days_vacant: number
  assigned_category: string
  minimum_threshold: number
  cap_applied: boolean
  days_in_pool: number
}

export interface RentIQResults {
  date: string
  current_occupancy: number
  total_units: number
  occupied_units: number
  target_occupancy: number
  target_occupied_units: number
  rentiq_pool_count: number
  units_needed_for_95: number
  rentiq_active: boolean
  rentiq_units: RentIQUnit[]
  thresholds: RentIQThreshold[]
}

export interface RentIQThreshold {
  config_key: string
  category_name: string
  min_rent: number
}

export class RentIQAnalytics {
  private static instance: RentIQAnalytics
  
  // Units excluded from RentIQ pool (special circumstances: renovation, reserved, etc.)
  private static readonly EXCLUDED_UNITS = ['704', '111', '312', '410', '510', '902', '311']
  
  /**
   * Check if a unit is a student unit (has - A or - B in the name)
   * Student units are always excluded from RentIQ calculations
   * Pattern: "XXX - A" or "XXX - B" (e.g., "114 - A", "120 - B")
   */
  private static isStudentUnit(unitName: string): boolean {
    const trimmed = unitName.trim()
    // Match " - A" or " - B" at end of string (case insensitive)
    return /\s-\s[AB]$/i.test(trimmed)
  }
  
  /**
   * Check if a unit should be excluded from RentIQ entirely
   * Combines static exclusion list and student unit check
   */
  private static isRentIQExcludedUnit(unitName: string): boolean {
    return this.EXCLUDED_UNITS.includes(unitName) || this.isStudentUnit(unitName)
  }
  
  static getInstance(): RentIQAnalytics {
    if (!RentIQAnalytics.instance) {
      RentIQAnalytics.instance = new RentIQAnalytics()
    }
    return RentIQAnalytics.instance
  }

  /**
   * Calculate RentIQ results for a given date (defaults to latest available date)
   */
  async calculateRentIQ(targetDate?: string): Promise<RentIQResults> {
    try {
      // If no date provided, use the latest available date
      if (!targetDate) {
        const latestDateResult = await prisma.masterTenantData.aggregate({
          _max: { snapshotDate: true }
        })
        targetDate = latestDateResult._max.snapshotDate?.toISOString().split('T')[0] || new Date().toISOString().split('T')[0]
      }
      
      console.log(`[RENTIQ] Calculating RentIQ for date: ${targetDate}`)
      
      // TIMEZONE FIX: Use date range query to handle timezone offsets in database
      // Eastern date "2025-10-10" may be stored as 2025-10-10T04:00:00.000Z in DB
      const startOfDay = new Date(`${targetDate}T00:00:00.000Z`)
      const endOfDay = new Date(`${targetDate}T23:59:59.999Z`)
      
      // Get master CSV data with actual unit type information from master.csv (unique units only)
      let masterDataRaw = await prisma.masterTenantData.findMany({
        where: {
          snapshotDate: {
            gte: startOfDay,
            lte: endOfDay
          }
        },
        // We'll join manually since Prisma doesn't have the relation set up
        orderBy: {
          unitCode: 'asc'
        }
      })
      
      // FALLBACK: If no data found for exact date, use latest available snapshot
      if (masterDataRaw.length === 0) {
        console.warn(`[RENTIQ] No data found for ${targetDate}, using latest snapshot`)
        masterDataRaw = await prisma.masterTenantData.findMany({
          orderBy: {
            snapshotDate: 'desc'
          },
          take: 182 // Get latest snapshot (all units)
        })
      }

      // V18.3.0: Use smart row selection to join CSV data (handles duplicate unit statuses)
      const masterCsvData = await prisma.masterCsvData.findMany()
      
      // Group CSV rows by unit (may have multiple rows per unit: Vacant + Future)
      const csvUnitGroups = new Map<string, typeof masterCsvData>()
      for (const row of masterCsvData) {
        if (!csvUnitGroups.has(row.unit)) {
          csvUnitGroups.set(row.unit, [])
        }
        csvUnitGroups.get(row.unit)!.push(row)
      }
      
      // Status priority: Future/Notice/Current > Vacant (match UnifiedAnalytics)
      // Use prefix matching to handle variants like "Notice Unrented"
      const getStatusPriority = (status: string | null | undefined): number => {
        const s = (status || '').toLowerCase().trim()
        if (s.startsWith('future')) return 4  // "Future", etc.
        if (s.startsWith('notice')) return 3  // "Notice", "Notice Unrented", etc.
        if (s.startsWith('current')) return 2 // "Current"
        if (s.startsWith('vacant')) return 1  // "Vacant"
        return 0 // Unknown status
      }
      
      // Select best row per unit (highest priority status)
      const csvDataMap = new Map<string, typeof masterCsvData[0]>()
      for (const [unit, rows] of csvUnitGroups) {
        const selectedRow = rows.reduce((best, current) => {
          const bestPriority = getStatusPriority(best.tenantStatus)
          const currentPriority = getStatusPriority(current.tenantStatus)
          return currentPriority > bestPriority ? current : best
        })
        csvDataMap.set(unit, selectedRow)
      }
      
      // Map raw data and EXCLUDE student units and other RentIQ-excluded units
      const masterDataAll = masterDataRaw.map(mtd => ({
        'Unit': mtd.unitCode,
        // Use smart-selected CSV row (highest priority status: Future > Notice > Current > Vacant)
        'Tenant Status': csvDataMap.get(mtd.unitCode)?.tenantStatus || (mtd.isOccupied ? 'Current' : 'Vacant'),
        'Monthly Rent': mtd.mrrAmount,
        'Market Rent': csvDataMap.get(mtd.unitCode)?.marketRent || mtd.marketRent,
        'Unit Type': csvDataMap.get(mtd.unitCode)?.unitType || null,
        'Days Vacant': csvDataMap.get(mtd.unitCode)?.daysVacant || 0
      }))

      // Filter out student units and other excluded units BEFORE any calculations
      const studentUnits = masterDataAll.filter(row => RentIQAnalytics.isStudentUnit(row['Unit']))
      const otherExcludedUnits = masterDataAll.filter(row => 
        RentIQAnalytics.EXCLUDED_UNITS.includes(row['Unit']) && !RentIQAnalytics.isStudentUnit(row['Unit'])
      )
      const masterData = masterDataAll.filter(row => !RentIQAnalytics.isRentIQExcludedUnit(row['Unit']))

      console.log(`[RENTIQ] Found ${masterDataAll.length} total units for ${targetDate}`)
      if (studentUnits.length > 0) {
        console.log(`[RENTIQ] 🎓 Excluded ${studentUnits.length} student units: ${[...new Set(studentUnits.map(u => u['Unit']))].join(', ')}`)
      }
      if (otherExcludedUnits.length > 0) {
        console.log(`[RENTIQ] 🚫 Excluded ${otherExcludedUnits.length} special units: ${[...new Set(otherExcludedUnits.map(u => u['Unit']))].join(', ')}`)
      }

      if (masterData.length === 0) {
        console.warn(`[RENTIQ] No master tenant data found for date: ${targetDate}`)
        throw new Error(`No master tenant data available for ${targetDate}`)
      }

      // Get total unique units from master.csv EXCLUDING student units and other excluded units
      // V18.3.0: Count DISTINCT units, not total rows (which includes duplicate statuses)
      const allUniqueUnits = csvUnitGroups.size || 182
      const excludedUnitCount = [...new Set([...studentUnits.map(u => u['Unit']), ...otherExcludedUnits.map(u => u['Unit'])])].length
      const totalUnits = allUniqueUnits - excludedUnitCount
      
      console.log(`[RENTIQ] Total units for RentIQ: ${totalUnits} (${allUniqueUnits} total - ${excludedUnitCount} excluded)`)

      // V18.3.0 SMART VACANCY DEDUPLICATION:
      // Group by unit to handle units with multiple status rows (e.g., Vacant + Future)
      const unitStatusMap = new Map<string, string[]>()
      for (const row of masterData) {
        const unit = row['Unit']
        if (!unitStatusMap.has(unit)) {
          unitStatusMap.set(unit, [])
        }
        unitStatusMap.get(unit)!.push(row['Tenant Status'])
      }
      
      // Check feature flag for smart vacancy deduplication
      const useSmartDedup = process.env.USE_SMART_VACANCY_DEDUPLICATION !== 'false'
      console.log(`[RENTIQ] Vacancy mode: ${useSmartDedup ? 'SMART deduplication (v18.3.0)' : 'LEGACY row-based'}`)
      
      // Identify truly vacant units (excluding units with Future/Notice tenants)
      const excludedUnits: string[] = []
      const trulyVacantUnits: string[] = []
      
      for (const [unit, statuses] of unitStatusMap) {
        const uniqueStatuses = [...new Set(statuses)]
        const hasVacant = uniqueStatuses.some(s => s === 'Vacant')
        const hasNonVacant = uniqueStatuses.some(s => s !== 'Vacant' && s !== '')
        
        if (hasVacant && !hasNonVacant) {
          // Unit is ONLY vacant (no other statuses)
          trulyVacantUnits.push(unit)
        } else if (hasVacant && hasNonVacant && useSmartDedup) {
          // Unit has both Vacant and non-vacant rows - exclude from vacant count
          excludedUnits.push(unit)
          console.log(`[RENTIQ] 📋 Excluding ${unit} from vacant count (statuses: ${uniqueStatuses.join(', ')})`)
        }
      }
      
      // Filter masterData to get only truly vacant unit rows
      const vacantUnits = useSmartDedup 
        ? masterData.filter(row => trulyVacantUnits.includes(row['Unit']))
        : masterData.filter(row => row['Tenant Status'] === 'Vacant')
      
      // CRITICAL: Match UnifiedAnalytics logic - anything NOT truly vacant is considered occupied
      const occupiedUnits = masterData.filter(row => {
        if (useSmartDedup) {
          return !trulyVacantUnits.includes(row['Unit'])
        }
        return row['Tenant Status'] !== 'Vacant'
      })
      
      // Calculate basic occupancy metrics using all 182 units (family units are always occupied)
      const uniqueOccupiedUnits = [...new Set(occupiedUnits.map(r => r['Unit']))].length
      const currentOccupancy = (uniqueOccupiedUnits / totalUnits) * 100
      
      // Target is 95% occupancy
      const targetOccupancy = 95
      const targetOccupiedUnits = Math.ceil((targetOccupancy / 100) * totalUnits) // 173 units
      const allowedVacantUnits = totalUnits - targetOccupiedUnits // 9 units
      
      console.log(`[RENTIQ] Vacancy breakdown: ${trulyVacantUnits.length} truly vacant (${excludedUnits.length} excluded with Future/Notice)`)
      
      // Calculate RentIQ Pool: vacant units minus allowed vacant units (9)
      const rentiqPoolCount = Math.max(0, trulyVacantUnits.length - allowedVacantUnits)
      const unitsNeededFor95 = Math.max(0, targetOccupiedUnits - uniqueOccupiedUnits)
      
      // RentIQ is active if pool count > 0
      const rentiqActive = rentiqPoolCount > 0
      
      console.log(`[RENTIQ] Occupancy: ${uniqueOccupiedUnits}/${totalUnits} (${currentOccupancy.toFixed(1)}%), Vacant: ${trulyVacantUnits.length}, Pool: ${rentiqPoolCount}`)

      // Calculate pricing for RentIQ pool units
      const rentiqUnits: RentIQUnit[] = []
      
      if (rentiqActive && vacantUnits.length > 0) {
        // CRITICAL: Sort vacant units by HIGHEST days vacant first (descending order)
        // This ensures units with longest vacancy get priority for progressive discounting
        // NOTE: Student units and other excluded units already filtered out in masterData
        const poolUnits = vacantUnits
          .sort((a, b) => (b['Days Vacant'] || 0) - (a['Days Vacant'] || 0))
          .slice(0, rentiqPoolCount)
        
        const thresholdsArray = await this.getThresholdsArray()
        
        for (const unit of poolUnits) {
          const marketRent = this.parseRent(unit['Market Rent'])
          const monthlyRent = this.parseRent(unit['Monthly Rent'])
          const daysVacant = unit['Days Vacant'] || 0
          const unitType = unit['Unit Type']
          
          // Assign category based on unit type or market rent
          const assignedCategory = this.assignCategoryFromUnitType(unitType, marketRent, thresholdsArray)
          const minimumThreshold = this.getThresholdForCategory(assignedCategory, thresholdsArray)
          
          // Calculate progressive suggested rent
          const { suggestedRent, tier, capApplied } = this.calculateProgressiveSuggestedRent(
            marketRent || 0,
            daysVacant,
            minimumThreshold
          )
          
          rentiqUnits.push({
            unit: unit.Unit || '',
            tenant_status: unit['Tenant Status'] || '',
            monthly_rent: monthlyRent,
            market_rent: marketRent,
            unit_type: unitType,
            suggested_new_rent: suggestedRent,
            pricing_tier: tier,
            days_vacant: daysVacant,
            assigned_category: assignedCategory,
            minimum_threshold: minimumThreshold,
            cap_applied: capApplied,
            days_in_pool: daysVacant
          })
        }
      }

      const results: RentIQResults = {
        date: targetDate!,
        current_occupancy: Math.round(currentOccupancy * 100) / 100,
        total_units: totalUnits,
        occupied_units: uniqueOccupiedUnits,
        target_occupancy: targetOccupancy,
        target_occupied_units: targetOccupiedUnits,
        rentiq_pool_count: rentiqPoolCount,
        units_needed_for_95: unitsNeededFor95,
        rentiq_active: rentiqActive,
        rentiq_units: rentiqUnits,
        thresholds: await this.getThresholdsArray() // Load actual thresholds
      }

      // Store results in database for caching
      await this.storeRentIQResults(results)
      
      console.log(`[RENTIQ] ✅ Calculated RentIQ: ${rentiqPoolCount} units in pool, active: ${rentiqActive}`)
      return results

    } catch (error) {
      console.error(`[RENTIQ] Error calculating RentIQ:`, error)
      throw error
    }
  }

  /**
   * Assign category from unit type or fallback to market rent
   */
  private assignCategoryFromUnitType(unitType: string | null, marketRent: number | null, thresholds: RentIQThreshold[]): string {
    if (unitType) {
      const category = this.mapUnitTypeToCategory(unitType)
      if (category) {
        return category
      }
    }
    
    // Fallback to market rent based assignment
    return this.assignCategoryByMarketRent(marketRent || 0, thresholds)
  }

  /**
   * Map unit types to RentIQ categories using token-based normalization
   * to prevent substring matching bugs (e.g., "unfurnished" containing "furnished")
   */
  private mapUnitTypeToCategory(unitType: string): string | null {
    const type = unitType.toLowerCase()
    
    // Normalize tokens: split on non-letters and create a set for exact matching
    const tokens = new Set(type.split(/[^a-z]+/).filter(t => t.length > 0))
    
    // Determine furnished status with explicit token matching (no substring issues)
    const isUnfurnished = tokens.has('unfurnished')
    const isFurnished = tokens.has('furnished') && !isUnfurnished
    
    // Determine property family
    const isMonaco = tokens.has('monaco')
    const isCapri = tokens.has('capri')
    const isMartinique = tokens.has('martinique')
    const isBarcelona = tokens.has('barcelona')
    const isNautica = tokens.has('nautica')
    const isPortofino = tokens.has('portofino')
    const isStudent = tokens.has('student')
    
    // Special case: Student units
    if (isStudent) {
      return 'Shared 1 BD Furnished'
    }
    
    // Basic: Monaco, Martinique, Capri
    if (isMonaco || isMartinique || isCapri) {
      if (isFurnished) return 'Basic-Furnished'
      if (isUnfurnished) return 'Basic-Unfurnished'
      // Log unmapped basic variant
      console.warn(`[RENTIQ] Basic unit type "${unitType}" has no furnished/unfurnished token - falling back to market rent`)
      return null
    }
    
    // Upgraded: Barcelona (intermediate tier)
    if (isBarcelona) {
      if (isFurnished) return 'Upgraded-Furnished'
      if (isUnfurnished) return 'Upgraded-Unfurnished'
      // Log unmapped upgraded variant
      console.warn(`[RENTIQ] Upgraded unit type "${unitType}" has no furnished/unfurnished token - falling back to market rent`)
      return null
    }
    
    // Premium: Nautica, Portofino
    if (isNautica || isPortofino) {
      if (isFurnished) return 'Premium-Furnished'
      if (isUnfurnished) return 'Premium-Unfurnished'
      // Log unmapped premium variant
      console.warn(`[RENTIQ] Premium unit type "${unitType}" has no furnished/unfurnished token - falling back to market rent`)
      return null
    }
    
    // Log completely unknown unit type for operational visibility
    console.warn(`[RENTIQ] Unknown unit type "${unitType}" - falling back to market rent categorization`)
    return null
  }

  /**
   * Assign category based on exact market rent match (fallback)
   * Updated tier structure: Basic (Monaco/Martinique/Capri), Upgraded (Barcelona), Premium (Nautica/Portofino), Shared (Student)
   */
  private assignCategoryByMarketRent(marketRent: number, thresholds: RentIQThreshold[]): string {
    // Exact market rent to category mapping
    const categoryMapping: { [key: number]: string } = {
      1500: 'Shared 1 BD Furnished',
      1990: 'Basic-Unfurnished',      // Monaco/Martinique unfurnished
      2000: 'Basic-Unfurnished',      // Capri unfurnished
      2020: 'Upgraded-Unfurnished',   // Barcelona unfurnished
      2120: 'Upgraded-Furnished',     // Barcelona furnished
      2220: 'Premium-Unfurnished',    // Nautica/Portofino unfurnished
      2240: 'Basic-Furnished',        // Monaco/Martinique/Capri furnished
      2370: 'Basic-Furnished',        // High-end Basic furnished
      2570: 'Premium-Furnished'       // Nautica/Portofino furnished
    }

    const category = categoryMapping[marketRent]
    if (category) {
      return category
    }

    // Fallback: assign based on rent ranges
    if (marketRent >= 2500) return 'Premium-Furnished'
    if (marketRent >= 2200) return 'Premium-Unfurnished'
    if (marketRent >= 2100) return 'Upgraded-Furnished'
    if (marketRent >= 2000) return 'Upgraded-Unfurnished'
    if (marketRent >= 1900) return 'Basic-Furnished'
    if (marketRent >= 1700) return 'Basic-Unfurnished'
    return 'Shared 1 BD Furnished'
  }

  /**
   * Get threshold for category
   */
  private getThresholdForCategory(category: string, thresholds: RentIQThreshold[]): number {
    const threshold = thresholds.find(t => t.category_name === category)
    return threshold?.min_rent || 1500 // Default minimum
  }

  /**
   * Calculate progressive suggested rent with tier-based discounting
   */
  private calculateProgressiveSuggestedRent(
    marketRent: number,
    daysVacant: number,
    minimumThreshold: number
  ): { suggestedRent: number; tier: 'Tier 1' | 'Tier 2' | 'Tier 3'; capApplied: boolean } {
    
    // Determine tier based on market rent
    let tier: 'Tier 1' | 'Tier 2' | 'Tier 3'
    if (marketRent >= 2300) {
      tier = 'Tier 1'
    } else if (marketRent >= 1800) {
      tier = 'Tier 2'
    } else {
      tier = 'Tier 3'
    }

    // Progressive discount based on days vacant
    let discount = 0
    
    if (daysVacant <= 6) {
      // Days 1-6
      discount = tier === 'Tier 1' ? 0.15 : tier === 'Tier 2' ? 0.10 : 0.00
    } else if (daysVacant <= 13) {
      // Days 7-13
      discount = tier === 'Tier 1' ? 0.20 : tier === 'Tier 2' ? 0.15 : 0.05
    } else if (daysVacant <= 20) {
      // Days 14-20
      discount = tier === 'Tier 1' ? 0.25 : tier === 'Tier 2' ? 0.20 : 0.10
    } else {
      // Days 21+
      discount = tier === 'Tier 1' ? 0.30 : tier === 'Tier 2' ? 0.25 : 0.15
    }

    // Calculate suggested rent
    const discountedRent = Math.round(marketRent * (1 - discount))
    
    // Apply minimum threshold cap
    const suggestedRent = Math.max(discountedRent, minimumThreshold)
    const capApplied = suggestedRent > discountedRent

    return { suggestedRent, tier, capApplied }
  }

  /**
   * Parse rent string to number
   */
  private parseRent(rentStr: any): number | null {
    if (!rentStr) return null
    
    const cleanStr = String(rentStr).replace(/[$,\s]/g, '')
    const num = parseFloat(cleanStr)
    
    return isNaN(num) ? null : num
  }

  /**
   * Store RentIQ results in database
   */
  private async storeRentIQResults(results: RentIQResults): Promise<void> {
    try {
      const key = `rentiq_results_${results.date}`
      
      // Store in a simple key-value table
      await prisma.kvStore.upsert({
        where: { key },
        update: {
          value: JSON.stringify(results),
          updatedAt: new Date()
        },
        create: {
          key,
          value: JSON.stringify(results),
          updatedAt: new Date()
        }
      })

      console.log(`[RENTIQ] Stored results for ${results.date}`)
    } catch (error) {
      console.warn(`[RENTIQ] Failed to store results:`, error)
      // Non-critical error - don't fail the calculation
    }
  }

  /**
   * Get latest RentIQ results from cache
   */
  async getLatestRentIQResults(): Promise<RentIQResults | null> {
    try {
      // Get the most recent RentIQ results
      const result = await prisma.kvStore.findFirst({
        where: {
          key: {
            startsWith: 'rentiq_results_'
          }
        },
        orderBy: {
          updatedAt: 'desc'
        }
      })

      if (result) {
        return JSON.parse(result.value) as RentIQResults
      }
      
      return null
    } catch (error) {
      console.warn(`[RENTIQ] Failed to get cached results:`, error)
      return null
    }
  }

  /**
   * Get RentIQ results for a specific date
   */
  async getRentIQResults(date: string): Promise<RentIQResults | null> {
    try {
      const key = `rentiq_results_${date}`
      
      const result = await prisma.kvStore.findUnique({
        where: { key }
      })

      if (result) {
        return JSON.parse(result.value) as RentIQResults
      }
      
      return null
    } catch (error) {
      console.warn(`[RENTIQ] Failed to get results for ${date}:`, error)
      return null
    }
  }

  /**
   * Get threshold settings with backward-compatible Upgraded tier addition
   */
  async getThresholds(): Promise<Record<string, number>> {
    try {
      // Try to load from database
      const stored = await prisma.kvStore.findUnique({
        where: { key: 'rentiq_thresholds' }
      })

      if (stored?.value) {
        try {
          const parsed = JSON.parse(stored.value)
          
          // Backward compatibility: Add Upgraded tier keys if missing
          const migrated: Record<string, number> = { ...parsed }
          let hasMigration = false
          
          // Add Upgraded-Unfurnished if missing (default: between Basic and Premium)
          if (!migrated['min_upgraded_unfurnished']) {
            migrated['min_upgraded_unfurnished'] = 1850
            hasMigration = true
            console.log(`[RENTIQ] 🔄 Adding missing Upgraded threshold: min_upgraded_unfurnished = 1850`)
          }
          
          // Add Upgraded-Furnished if missing
          if (!migrated['min_upgraded_furnished']) {
            migrated['min_upgraded_furnished'] = 2050
            hasMigration = true
            console.log(`[RENTIQ] 🔄 Adding missing Upgraded threshold: min_upgraded_furnished = 2050`)
          }
          
          // If migration occurred, save updated thresholds
          if (hasMigration) {
            await this.updateThresholds(migrated)
            console.log(`[RENTIQ] ✅ Migrated thresholds to include Upgraded tier`)
          }
          
          console.log(`[RENTIQ] ✅ Loaded thresholds from database:`, migrated)
          return migrated
        } catch (error) {
          console.warn('[RENTIQ] Failed to parse stored thresholds, using defaults:', error)
        }
      }
    } catch (error) {
      console.warn('[RENTIQ] Failed to load thresholds from database, using defaults:', error)
    }

    // Return updated defaults with Upgraded tier (Barcelona units)
    const defaults = {
      'min_basic_unfurnished': 1700,
      'min_basic_furnished': 1900,
      'min_upgraded_unfurnished': 1850,
      'min_upgraded_furnished': 2050,
      'min_premium_unfurnished': 2100,
      'min_premium_furnished': 2250,
      'min_student_unit': 1500
    }

    console.log(`[RENTIQ] Using default thresholds (Basic/Upgraded/Premium/Shared structure)`)
    return defaults
  }

  /**
   * Update threshold settings
   */
  async updateThresholds(thresholds: Record<string, number>): Promise<void> {
    try {
      const thresholdsJson = JSON.stringify(thresholds)
      
      await prisma.kvStore.upsert({
        where: { key: 'rentiq_thresholds' },
        create: {
          key: 'rentiq_thresholds',
          value: thresholdsJson,
          updatedAt: new Date()
        },
        update: {
          value: thresholdsJson,
          updatedAt: new Date()
        }
      })
      
      console.log(`[RENTIQ] ✅ Saved ${Object.keys(thresholds).length} thresholds to database:`, thresholds)
    } catch (error) {
      console.error('[RENTIQ] ❌ Failed to save thresholds:', error)
      throw new Error(`Failed to save thresholds: ${error instanceof Error ? error.message : 'Unknown error'}`)
    }
  }

  /**
   * Get threshold settings as array (for internal use)
   */
  private async getThresholdsArray(): Promise<RentIQThreshold[]> {
    const thresholdRecord = await this.getThresholds()
    
    const thresholds: RentIQThreshold[] = []
    
    for (const [key, value] of Object.entries(thresholdRecord)) {
      const categoryName = key.replace('min_', '').split('_').map(word => 
        word.charAt(0).toUpperCase() + word.slice(1)
      ).join('-')
      
      thresholds.push({
        config_key: key,
        category_name: categoryName,
        min_rent: value
      })
    }
    
    return thresholds
  }
}